// Background service worker
//
// IMPORTANT:
// - The extension can work with local Desktop App (default) OR with a remote VPS API.
// - Configure API base via chrome.storage.local key: `apiUrl` (e.g. https://api.example.com)
// - WebSocket URL is derived from apiUrl (wss/ws + same host, port 9876 by default, path '/ws')
const DEFAULT_API_URL = 'https://ai-lead-scraper.lazysoft.pl';
const DEFAULT_WS_PORT = 9876;
const DEFAULT_WS_PATH = '/ws';
let ws = null;
let reconnectTimer = null;

chrome.runtime.onInstalled.addListener(() => {
  console.log('AI Lead Scraper Extension installed');
  
  // Set default settings
  chrome.storage.local.set({
    apiUrl: DEFAULT_API_URL,
    autoScrape: false,
    notificationsEnabled: true
  });
  
  // Connect to WebSocket server
  connectWebSocket();
});

function deriveWsUrl(apiUrl) {
  try {
    const u = new URL(apiUrl || DEFAULT_API_URL);
    
    // For desktop app: prefer 127.0.0.1 WebSocket (port 9876, no path)
    // Desktop app WebSocket server doesn't use /ws path
    // Use 127.0.0.1 instead of localhost for better Windows compatibility
    if (u.hostname === 'localhost' || u.hostname === '127.0.0.1' || u.hostname.includes('localhost')) {
      return `ws://127.0.0.1:${DEFAULT_WS_PORT}`;
    }
    
    // For VPS: use wss with /ws path
    const wsProtocol = u.protocol === 'https:' ? 'wss:' : 'ws:';
    return `${wsProtocol}//${u.hostname}:${DEFAULT_WS_PORT}${DEFAULT_WS_PATH}`;
  } catch (e) {
    // Fallback: always try localhost for desktop app
    return `ws://localhost:${DEFAULT_WS_PORT}`;
  }
}

function connectWebSocket() {
  try {
    // Якщо є активний конект — не створюємо другий
    if (ws && (ws.readyState === WebSocket.OPEN || ws.readyState === WebSocket.CONNECTING)) {
      return;
    }

    chrome.storage.local.get(['apiUrl', 'wsUrl'], (res) => {
      // Priority: explicit wsUrl > localhost (for desktop) > derived from apiUrl
      let wsUrl = res.wsUrl;
      
        if (!wsUrl) {
          // For desktop app: ALWAYS try 127.0.0.1 first (desktop app runs locally)
          // Desktop app WebSocket runs on 127.0.0.1:9876 (no path)
          wsUrl = `ws://127.0.0.1:${DEFAULT_WS_PORT}`;
          console.log('[Extension] Using 127.0.0.1 WebSocket for desktop app:', wsUrl);
        }
      
      console.log('[Extension] Connecting to WebSocket:', wsUrl);
      ws = new WebSocket(wsUrl);
    
      ws.onopen = () => {
        console.log('Connected to backend via WebSocket:', wsUrl);
        if (reconnectTimer) {
          clearTimeout(reconnectTimer);
          reconnectTimer = null;
        }
        
        // Send connection message
        ws.send(JSON.stringify({
          action: 'ping',
          extension_id: chrome.runtime.id
        }));
      };
    
      ws.onmessage = async (event) => {
        try {
          const data = JSON.parse(event.data);
          
          if (data.action === 'scrape_requested') {
            // Trigger scraping on current page
            const [tab] = await chrome.tabs.query({ active: true, currentWindow: true });
            if (tab) {
              chrome.tabs.sendMessage(tab.id, { action: 'scrape' });
            }
          } else if (data.action === 'pong') {
            // Heartbeat response
            console.log('Heartbeat received');
          } else if (data.action === 'contact_added') {
            // Confirmation from backend (stored / rejected)
            const batchId = data.batch_id || null;
            const result = data.result || {};
            const ok = !!result.success;
            const err = result.error || null;

            if (batchId) {
              chrome.storage.local.get(['active_scrape_batch'], (res) => {
                const b = res.active_scrape_batch;
                if (b && b.id === batchId) {
                  b.confirmed = (b.confirmed || 0) + 1;
                  if (ok) b.saved = (b.saved || 0) + 1;
                  else b.failed = (b.failed || 0) + 1;
                  if (!ok && err) b.last_error = err;
                  b.updated_at = Date.now();
                  chrome.storage.local.set({ active_scrape_batch: b });
                }
              });
            } else {
              // No batch: still keep a last error if needed
              if (!ok && err) {
                chrome.storage.local.set({ last_scrape_error: err });
              }
            }
          }
        } catch (error) {
          console.error('Error processing WebSocket message:', error);
        }
      };
    
      ws.onerror = (error) => {
        console.error('WebSocket error:', error);
        // If local connection failed, try VPS as fallback (only if explicitly configured)
        const isLocalConn = wsUrl.includes('localhost') || wsUrl.includes('127.0.0.1');
        if (isLocalConn && !res.wsUrl) {
          const apiUrl = res.apiUrl || DEFAULT_API_URL;
          // Only try VPS if apiUrl is explicitly set to VPS (not default/local)
          if (apiUrl && !apiUrl.includes('localhost') && !apiUrl.includes('127.0.0.1')) {
            const vpsWsUrl = deriveWsUrl(apiUrl);
            if (vpsWsUrl !== wsUrl) {
              console.log('[Extension] Local connection failed, will try VPS on reconnect:', vpsWsUrl);
              // Store VPS URL for next reconnect attempt
              chrome.storage.local.set({ wsUrl: vpsWsUrl });
            }
          } else {
            console.log('[Extension] Local connection failed - desktop app may not be running');
          }
        }
      };
      
      ws.onclose = (event) => {
        console.log('WebSocket disconnected, code:', event.code, 'reason:', event.reason);
        
        // If local connection failed, clear wsUrl to try again
        const isLocalConnClose = wsUrl.includes('localhost') || wsUrl.includes('127.0.0.1');
        if (isLocalConnClose && event.code === 1006) {
          // Connection failed - clear any stored wsUrl to retry local
          chrome.storage.local.remove(['wsUrl'], () => {
            console.log('[Extension] Cleared wsUrl, will retry localhost on reconnect');
          });
        }
        
        // Плануємо один reconnect через 5 секунд, без нескінченних інтервалів
        if (!reconnectTimer) {
          reconnectTimer = setTimeout(() => {
            reconnectTimer = null;
            connectWebSocket();
          }, 5000);
        }
      };
    
    });
  } catch (error) {
    console.error('Failed to connect WebSocket:', error);
    // Одноразова спроба перепідключення
    if (!reconnectTimer) {
      reconnectTimer = setTimeout(() => {
        reconnectTimer = null;
        connectWebSocket();
      }, 5000);
    }
  }
}

// If apiUrl changes, reconnect WS to the new backend.
chrome.storage.onChanged.addListener((changes, areaName) => {
  if (areaName !== 'local') return;
  if (!changes.apiUrl) return;
  try {
    if (ws) ws.close();
  } catch (e) {}
  // connectWebSocket() will be triggered by ws.onclose, but call it to speed up.
  connectWebSocket();
});

// Listen for messages from content script
chrome.runtime.onMessage.addListener((request, sender, sendResponse) => {
  if (request.action === 'scraped_contact') {
    // Get user_id from storage (set by desktop app)
    chrome.storage.local.get(['user_id', 'auth_token', 'api_key'], (result) => {
      const userId = result.user_id || 1; // Fallback to 1 if not set
      const apiKey = result.auth_token || result.api_key || null;
      const batchId = request.batch_id || null;
      
      // Send scraped contact to desktop app via WebSocket
      if (ws && ws.readyState === WebSocket.OPEN) {
        ws.send(JSON.stringify({
          action: 'add_contact',
          user_id: userId,
          api_key: apiKey,
          batch_id: batchId,
          data: request.contact
        }));
        // Track that we sent something for this batch (best-effort)
        if (batchId) {
          chrome.storage.local.get(['active_scrape_batch'], (res) => {
            const b = res.active_scrape_batch;
            if (b && b.id === batchId) {
              b.sent = (b.sent || 0) + 1;
              b.updated_at = Date.now();
              chrome.storage.local.set({ active_scrape_batch: b });
            }
          });
        }
        sendResponse({ success: true });
      } else {
        if (batchId) {
          chrome.storage.local.get(['active_scrape_batch'], (res) => {
            const b = res.active_scrape_batch;
            if (b && b.id === batchId) {
              b.last_error = 'WebSocket not connected';
              b.updated_at = Date.now();
              chrome.storage.local.set({ active_scrape_batch: b });
            }
          });
        }
        sendResponse({ success: false, error: 'WebSocket not connected' });
      }
    });
    return true; // Keep channel open for async response
  }
  
  if (request.action === 'ping') {
    sendResponse({ status: ws && ws.readyState === WebSocket.OPEN ? 'connected' : 'disconnected' });
  }
  
  if (request.action === 'restart_websocket') {
    console.log('[Extension] Restarting WebSocket connection...');
    // Close existing connection
    if (ws) {
      try {
        ws.close();
      } catch (e) {
        console.error('Error closing WebSocket:', e);
      }
      ws = null;
    }
    // Clear reconnect timer
    if (reconnectTimer) {
      clearTimeout(reconnectTimer);
      reconnectTimer = null;
    }
    // Reconnect immediately
    setTimeout(() => {
      connectWebSocket();
    }, 500);
    sendResponse({ success: true, message: 'WebSocket restart initiated' });
  }
  
  // Allow desktop app to set user_id and API key
  if (request.action === 'set_user_id') {
    chrome.storage.local.set({ user_id: request.user_id }, () => {
      sendResponse({ success: true });
    });
    return true;
  }
  
  // Allow desktop app to set API key
  if (request.action === 'set_api_key') {
    chrome.storage.local.set({ 
      api_key: request.api_key,
      auth_token: request.api_key  // Also store as auth_token for compatibility
    }, () => {
      sendResponse({ success: true });
    });
    return true;
  }
  
  // Allow desktop app to set full user data (user_id + api_key + email)
  if (request.action === 'set_user_data') {
    chrome.storage.local.set({ 
      user_id: request.user_id,
      api_key: request.api_key,
      auth_token: request.api_key,
      user_email: request.user_email || null
    }, () => {
      sendResponse({ success: true });
    });
    return true;
  }
});

// Keep service worker alive
setInterval(() => {
  chrome.storage.local.get('apiUrl', () => {
    // Keep alive
  });
}, 20000);

