const DEFAULT_API_BASE = 'https://ai-lead-scraper.lazysoft.pl';
let API_BASE = DEFAULT_API_BASE;

async function loadApiBase() {
  return new Promise((resolve) => {
    try {
      chrome.storage.local.get(['apiUrl'], (result) => {
        API_BASE = result.apiUrl || DEFAULT_API_BASE;
        resolve(API_BASE);
      });
    } catch (e) {
      API_BASE = DEFAULT_API_BASE;
      resolve(API_BASE);
    }
  });
}

let currentUserId = null;
let authToken = null;
let userCredits = 0;
let isLifetime = false;
let userEmail = null;
let activeBatchId = null;

function loadStoredAuth(callback) {
  chrome.storage.local.get(['user_id', 'auth_token', 'api_key', 'user_email'], (result) => {
    currentUserId = result.user_id || null;
    // Try both api_key and auth_token for compatibility
    authToken = result.api_key || result.auth_token || null;
    userEmail = result.user_email || null;

    const accountEmailEl = document.getElementById('account-email');
    const accountForm = document.getElementById('account-form');

    if (currentUserId && userEmail) {
      accountEmailEl.textContent = userEmail;
      if (accountForm) {
        accountForm.style.display = 'none';
      }
    } else {
      accountEmailEl.textContent = 'Not connected';
      if (accountForm) {
        accountForm.style.display = 'block';
      }
    }

    if (callback) callback();
  });
}

// Check connection to desktop app
async function checkConnection() {
  const statusIndicator = document.querySelector('.status-indicator');
  const statusText = document.getElementById('status-text');
  const scrapeBtn = document.getElementById('scrape-btn');

  try {
    // Health check (HTTP)
    const controller = new AbortController();
    const t = setTimeout(() => controller.abort(), 2500);
    const response = await fetch(`${API_BASE}/api/health`, { method: 'GET', signal: controller.signal });
    clearTimeout(t);
    
    if (response.ok) {
      // WebSocket check (via background service worker)
      const wsOk = await new Promise((resolve) => {
        try {
          chrome.runtime.sendMessage({ action: 'ping' }, (resp) => {
            resolve(resp && resp.status === 'connected');
          });
        } catch (e) {
          resolve(false);
        }
      });

      statusIndicator.classList.remove('disconnected');
      statusIndicator.classList.add('connected');
      statusText.textContent = wsOk ? 'Connected to Desktop App' : 'Desktop App OK, WS not connected (restart app)';

      // Дозволяємо скрапінг тільки якщо є прив'язаний акаунт
      loadStoredAuth(async () => {
        scrapeBtn.disabled = !currentUserId || !wsOk;
        if (currentUserId) {
          await loadUserInfo();
        }
      });
    }
  } catch (error) {
    statusIndicator.classList.add('disconnected');
    statusText.textContent = 'Desktop App not running. Please launch it.';
    scrapeBtn.disabled = true;
  }
}

// Load user credits info
async function loadUserInfo() {
  if (!currentUserId) return;
  try {
    // Get API key from storage
    const storage = await new Promise((resolve) => {
      chrome.storage.local.get(['api_key', 'auth_token'], resolve);
    });
    const apiKey = storage.api_key || storage.auth_token;
    
    if (!apiKey) {
      console.error('No API key found in storage');
      return;
    }
    
    const response = await fetch(`${API_BASE}/api/user/info?user_id=${currentUserId}`, {
      headers: {
        'api-key': apiKey,
        'X-API-Key': apiKey,
        'Content-Type': 'application/json'
      }
    });
    
    if (!response.ok) {
      if (response.status === 401) {
        console.error('API key invalid or expired');
        // Clear invalid auth
        chrome.storage.local.remove(['api_key', 'auth_token', 'user_id']);
        return;
      }
      throw new Error(`HTTP ${response.status}: ${response.statusText}`);
    }
    
    const data = await response.json();
    
    currentUserId = data.user_id;
    isLifetime = data.is_lifetime;
    
    // Save user_id to chrome.storage for background script
    if (currentUserId) {
      chrome.storage.local.set({ user_id: currentUserId });
    }
    
    const creditsEl = document.getElementById('credits-remaining');
    if (isLifetime) {
      creditsEl.textContent = '∞ Unlimited';
      creditsEl.style.color = 'var(--accent-pink)';
    } else {
      userCredits = data.credits_balance || (data.credits?.contacts || 0);
      creditsEl.textContent = userCredits;
    }
  } catch (error) {
    console.error('Failed to load user info:', error);
  }
}

// Scrape current page
document.getElementById('scrape-btn').addEventListener('click', async () => {
  const scrapeBtn = document.getElementById('scrape-btn');
  const resultsBox = document.getElementById('results');
  const resultsText = document.getElementById('results-text');
  const resultsProgress = document.getElementById('results-progress');
  
  scrapeBtn.disabled = true;
  scrapeBtn.textContent = '⏳ Scraping...';
  resultsText.textContent = '';
  resultsBox.style.display = 'none';
  resultsBox.classList.remove('state-error', 'state-success');
  if (resultsProgress) resultsProgress.style.width = '0%';
  
  try {
    // Send message to content script
    const [tab] = await chrome.tabs.query({ active: true, currentWindow: true });
    
    activeBatchId = `batch_${Date.now()}_${Math.random().toString(16).slice(2)}`;
    chrome.storage.local.set({
      active_scrape_batch: {
        id: activeBatchId,
        expected: 0,
        sent: 0,
        confirmed: 0,
        saved: 0,
        failed: 0,
        last_error: null,
        updated_at: Date.now()
      }
    });

    chrome.tabs.sendMessage(tab.id, { action: 'scrape', batch_id: activeBatchId }, async (response) => {
      if (chrome.runtime.lastError) {
        resultsText.textContent = 'Error: Could not access page. Try refreshing.';
        resultsBox.style.display = 'block';
        resultsBox.classList.add('state-error');
        scrapeBtn.disabled = false;
        scrapeBtn.textContent = '📊 Scrape This Page';
        return;
      }

      // Показуємо результат скрапінгу від content-script
      if (!response || response.success === false) {
        resultsText.textContent = response?.error || 'Scraping failed. Please try again.';
        resultsBox.style.display = 'block';
        resultsBox.classList.add('state-error');
      } else {
        const found = response.contacts_found ?? 0;
        resultsText.textContent = `Found ${found} contact(s). Sending to Desktop App...`;
        resultsBox.style.display = 'block';
        document.getElementById('contact-count').textContent = found;
        resultsBox.classList.remove('state-error', 'state-success');

        // Initialize batch expected count so we can display completion.
        chrome.storage.local.get(['active_scrape_batch'], (res) => {
          const b = res.active_scrape_batch;
          if (b && b.id === activeBatchId) {
            b.expected = found;
            b.updated_at = Date.now();
            chrome.storage.local.set({ active_scrape_batch: b });
          }
        });
      }

      scrapeBtn.disabled = false;
      scrapeBtn.textContent = '📊 Scrape This Page';
    });
  } catch (error) {
    resultsText.textContent = `Error: ${error.message}`;
    resultsBox.style.display = 'block';
    resultsBox.classList.add('state-error');
    scrapeBtn.disabled = false;
    scrapeBtn.textContent = '📊 Scrape This Page';
  }
});

// Account connect / register-login flow
document.getElementById('account-form').addEventListener('submit', async (e) => {
  e.preventDefault();
  const emailInput = document.getElementById('acc-email');
  const passwordInput = document.getElementById('acc-password');
  const companyInput = document.getElementById('acc-company');
  const countryInput = document.getElementById('acc-country');
  const msgEl = document.getElementById('account-message');
  const btn = document.getElementById('connect-btn');

  const email = emailInput.value.trim();
  const password = passwordInput.value.trim();
  const company = companyInput.value.trim();
  const country = countryInput.value.trim();

  if (!email || !password || !company || !country) {
    msgEl.textContent = 'Please fill in all required fields.';
    return;
  }

  btn.disabled = true;
  msgEl.textContent = 'Connecting...';

  try {
    // Use the same auth API as Desktop App (PostgreSQL users)
    // Try register first
    let response = await fetch(`${API_BASE}/api/auth/register`, {
      method: 'POST',
      headers: { 'Content-Type': 'application/json' },
      body: JSON.stringify({
        email,
        password,
        company,
        country,
        tax_id: null
      })
    });

    if (!response.ok) {
      // If email already exists, fallback to login
      if (response.status === 400) {
        response = await fetch(`${API_BASE}/api/auth/login`, {
          method: 'POST',
          headers: { 'Content-Type': 'application/json' },
          body: JSON.stringify({ email, password })
        });
      }
    }

    if (!response.ok) {
      const errText = await response.text();
      throw new Error(errText || 'Failed to connect account');
    }

    const data = await response.json();
    currentUserId = data.user_id;
    // API key is used by Desktop app; keep it for compatibility
    authToken = data.api_key || null;
    userEmail = data.email || email;

    // Store API key with both keys for compatibility
    chrome.storage.local.set(
      { 
        user_id: currentUserId, 
        auth_token: authToken, 
        api_key: authToken,  // Also store as api_key for consistency
        user_email: userEmail 
      },
      () => {
        const accountEmailEl = document.getElementById('account-email');
        accountEmailEl.textContent = userEmail;
        document.getElementById('account-form').style.display = 'none';
        document.getElementById('scrape-btn').disabled = false;
      }
    );

    msgEl.textContent = 'Account connected successfully.';
    await loadUserInfo();
  } catch (error) {
    console.error('Account connect error:', error);
    msgEl.textContent = 'Failed to connect account. Please ensure desktop app is running.';
  } finally {
    btn.disabled = false;
  }
});

// Live progress updates from background (WebSocket confirmations)
chrome.storage.onChanged.addListener((changes, areaName) => {
  if (areaName !== 'local') return;
  if (!changes.active_scrape_batch) return;

  const batch = changes.active_scrape_batch.newValue;
  if (!batch || !batch.id) return;

  const resultsBox = document.getElementById('results');
  const resultsText = document.getElementById('results-text');
  const resultsProgress = document.getElementById('results-progress');

  const expected = batch.expected ?? 0;
  const confirmed = batch.confirmed ?? 0;
  const saved = batch.saved ?? 0;
  const failed = batch.failed ?? 0;
  const lastErr = batch.last_error;

  if (expected > 0) {
    const done = confirmed >= expected;
    const pct = Math.max(0, Math.min(100, Math.round((confirmed / expected) * 100)));
    if (resultsProgress) resultsProgress.style.width = `${pct}%`;
    resultsText.textContent =
      done
        ? `Done. Saved: ${saved}/${expected}. Failed: ${failed}${lastErr ? `. Last error: ${lastErr}` : ''}`
        : `Sending... Confirmed: ${confirmed}/${expected}. Saved: ${saved}. Failed: ${failed}${lastErr ? `. Last error: ${lastErr}` : ''}`;
    resultsBox.style.display = 'block';
    resultsBox.classList.toggle('state-success', done && failed === 0);
    resultsBox.classList.toggle('state-error', done && failed > 0);
  }
});

// Restart extension button
document.getElementById('restart-btn').addEventListener('click', async () => {
  const restartBtn = document.getElementById('restart-btn');
  const statusText = document.getElementById('status-text');
  
  restartBtn.disabled = true;
  restartBtn.textContent = '🔄 Restarting...';
  statusText.textContent = 'Restarting extension...';
  
  try {
    // Send message to background script to reconnect WebSocket
    chrome.runtime.sendMessage({ action: 'restart_websocket' }, (response) => {
      if (chrome.runtime.lastError) {
        console.error('Error restarting:', chrome.runtime.lastError);
        statusText.textContent = 'Error restarting. Please try again.';
      } else {
        statusText.textContent = 'Restarting...';
        // Wait a moment then check connection
        setTimeout(() => {
          checkConnection();
          restartBtn.disabled = false;
          restartBtn.textContent = '🔄 Restart';
        }, 2000);
      }
    });
  } catch (error) {
    console.error('Failed to restart:', error);
    statusText.textContent = 'Error restarting. Please try again.';
    restartBtn.disabled = false;
    restartBtn.textContent = '🔄 Restart';
  }
});

// Initialize
(async () => {
  await loadApiBase();
  loadStoredAuth(() => {
    checkConnection();
    setInterval(checkConnection, 10000); // Check every 10s
  });
})();

