const DEFAULT_API_BASE = 'https://ai-lead-scraper.lazysoft.pl';
let API_BASE = DEFAULT_API_BASE;

async function loadApiBase() {
  return new Promise((resolve) => {
    try {
      chrome.storage.local.get(['apiUrl'], (result) => {
        API_BASE = result.apiUrl || DEFAULT_API_BASE;
        resolve(API_BASE);
      });
    } catch (e) {
      API_BASE = DEFAULT_API_BASE;
      resolve(API_BASE);
    }
  });
}

// Tab switching
document.querySelectorAll('.auth-tab').forEach(tab => {
  tab.addEventListener('click', () => {
    const tabName = tab.dataset.tab;

    // Update active tab
    document.querySelectorAll('.auth-tab').forEach(t => t.classList.remove('active'));
    tab.classList.add('active');

    // Show corresponding form
    document.querySelectorAll('.auth-form').forEach(form => form.classList.remove('active'));
    document.getElementById(`${tabName}-form`).classList.add('active');
  });
});

// Login form
document.getElementById('loginForm').addEventListener('submit', async (e) => {
  e.preventDefault();
  await loadApiBase();

  const email = document.getElementById('login-email').value;
  const password = document.getElementById('login-password').value;
  const messageEl = document.getElementById('login-message');
  const submitBtn = e.target.querySelector('.btn-submit');

  submitBtn.disabled = true;
  submitBtn.textContent = 'Logging in...';
  messageEl.textContent = '';
  messageEl.className = 'error-message';

  try {
    const response = await fetch(`${API_BASE}/api/auth/login`, {
      method: 'POST',
      headers: { 'Content-Type': 'application/json' },
      body: JSON.stringify({ email, password })
    });

    const data = await response.json();

    if (response.ok) {
      // Save credentials to chrome storage
      await chrome.storage.local.set({
        user_id: data.user_id,
        user_email: data.email,
        api_key: data.api_key,
        credits_balance: data.credits_balance,
        is_lifetime: data.is_lifetime
      });

      messageEl.className = 'success-message';
      messageEl.textContent = 'Login successful! Redirecting...';

      // Redirect to main popup
      setTimeout(() => {
        window.location.href = 'popup.html';
      }, 1000);
    } else {
      messageEl.textContent = data.detail || 'Login failed';
    }
  } catch (error) {
    messageEl.textContent = 'Connection error. Is the desktop app running?';
  } finally {
    submitBtn.disabled = false;
    submitBtn.textContent = 'Login';
  }
});

// Register form
document.getElementById('registerForm').addEventListener('submit', async (e) => {
  e.preventDefault();
  await loadApiBase();

  const email = document.getElementById('reg-email').value;
  const password = document.getElementById('reg-password').value;
  const name = document.getElementById('reg-name').value;
  const company = document.getElementById('reg-company').value;
  const country = document.getElementById('reg-country').value;
  const messageEl = document.getElementById('register-message');
  const submitBtn = e.target.querySelector('.btn-submit');

  submitBtn.disabled = true;
  submitBtn.textContent = 'Creating account...';
  messageEl.textContent = '';
  messageEl.className = 'error-message';

  try {
    const response = await fetch(`${API_BASE}/api/auth/register`, {
      method: 'POST',
      headers: { 'Content-Type': 'application/json' },
      body: JSON.stringify({
        email,
        password,
        name: name || null,
        company: company || null,
        country: country || null
      })
    });

    const data = await response.json();

    if (response.ok) {
      // Save credentials to chrome storage
      await chrome.storage.local.set({
        user_id: data.user_id,
        user_email: data.email,
        api_key: data.api_key,
        credits_balance: data.credits_balance,
        is_lifetime: data.is_lifetime
      });

      messageEl.className = 'success-message';
      messageEl.textContent = 'Account created! You got 200 FREE credits. Redirecting...';

      // Redirect to main popup
      setTimeout(() => {
        window.location.href = 'popup.html';
      }, 1500);
    } else {
      messageEl.textContent = data.detail || 'Registration failed';
    }
  } catch (error) {
    messageEl.textContent = 'Connection error. Is the desktop app running?';
  } finally {
    submitBtn.disabled = false;
    submitBtn.textContent = 'Create Account';
  }
});

// Check if already logged in
chrome.storage.local.get(['user_id', 'api_key'], (result) => {
  if (result.user_id && result.api_key) {
    // Already logged in, redirect to main popup
    window.location.href = 'popup.html';
  }
});
