// Extract phone numbers (international formats, with country-aware heuristics)
function extractPhones() {
  const text = document.body.innerText;

  // Broad match for potential phone-like substrings
  const phoneRegex = /(\+?\d[\d()\-\s.]{6,}\d)/g;
  const textMatches = text.match(phoneRegex) || [];

  // Also check tel: links (usually найнадійніші джерела)
  const telLinks = Array.from(document.querySelectorAll('a[href^="tel:"]'))
    .map(link => link.href.replace(/^tel:/i, ''));

  const candidates = [...telLinks, ...textMatches];
  const seenDigits = new Set();
  const results = [];

  const countryPatterns = [
    // E.164 generic (+XXXXXXXX)
    { name: 'e164', test: (d, raw) => /^\+\d{8,15}$/.test(raw.replace(/[^\d+]/g, '')) },
    // USA / Canada
    { name: 'us_ca', test: d => /^(?:1)?\d{10}$/.test(d) },
    // UK
    { name: 'uk', test: d => /^(?:44|0)\d{9,10}$/.test(d) },
    // Germany
    { name: 'de', test: d => /^(?:49|0)\d{10,13}$/.test(d) },
    // France
    { name: 'fr', test: d => /^(?:33|0)\d{9}$/.test(d) },
    // Spain
    { name: 'es', test: d => /^(?:34|0)\d{9}$/.test(d) },
    // Italy
    { name: 'it', test: d => /^(?:39|0)\d{9,11}$/.test(d) },
    // Netherlands
    { name: 'nl', test: d => /^(?:31|0)\d{9}$/.test(d) },
    // Poland
    { name: 'pl', test: d => /^(?:48|0)\d{9}$/.test(d) },
    // Ukraine
    { name: 'ua', test: d => /^(?:380|0)\d{9}$/.test(d) },
  ];

  function looksLikePhone(raw) {
    const digits = raw.replace(/\D/g, '');

    // Відсікаємо індекси / ZIP коди та інший шум
    if (digits.length < 9 || digits.length > 15) return false;

    // Перевірка по відомим шаблонам країн
    if (countryPatterns.some(p => p.test(digits, raw))) return true;

    // Фолбек: достатньо довгий номер, який містить хоча б один телефонний символ
    if ((raw.includes('+') || raw.includes('(') || raw.includes(')') || raw.includes('-')) &&
        digits.length >= 9 && digits.length <= 15) {
      return true;
    }

    return false;
  }

  for (const candidate of candidates) {
    const raw = (candidate || '').trim();
    if (!raw) continue;

    if (!looksLikePhone(raw)) continue;

    const digits = raw.replace(/\D/g, '');
    if (seenDigits.has(digits)) continue;
    seenDigits.add(digits);
    results.push(raw);
  }

  return results;
}

