#!/usr/bin/env python3
"""
Simple script to generate placeholder icons for Chrome Extension
Requires PIL/Pillow: pip install Pillow
"""

try:
    from PIL import Image, ImageDraw, ImageFont
except ImportError:
    print("Pillow not installed. Install with: pip install Pillow")
    exit(1)

def create_icon(size, output_path):
    """Create a simple icon with gradient background"""
    # Create image with black background
    img = Image.new('RGB', (size, size), color='#000000')
    draw = ImageDraw.Draw(img)
    
    # Draw gradient-like effect (simple version)
    # Draw green circle
    margin = size // 8
    draw.ellipse([margin, margin, size - margin, size - margin], 
                 fill='#00FF00', outline='#FF1493', width=2)
    
    # Add text "AI" in center
    try:
        font_size = size // 3
        font = ImageFont.truetype("/usr/share/fonts/truetype/dejavu/DejaVuSans-Bold.ttf", font_size)
    except:
        # Fallback to default font
        font = ImageFont.load_default()
    
    text = "AI"
    bbox = draw.textbbox((0, 0), text, font=font)
    text_width = bbox[2] - bbox[0]
    text_height = bbox[3] - bbox[1]
    position = ((size - text_width) // 2, (size - text_height) // 2 - text_height // 4)
    
    draw.text(position, text, fill='#000000', font=font)
    
    img.save(output_path)
    print(f"Created {output_path}")

if __name__ == "__main__":
    sizes = [16, 48, 128]
    for size in sizes:
        create_icon(size, f"icons/icon{size}.png")
    print("All icons generated!")

